; ─────────────────────────────────────────────────────────
; Launcher.v2.ahk  (AutoHotkey v2)
; מטרה:
;   - להוריד worker AHK + JSON
;   - להוסיף ל-JSON נתוני הפעלה (שם/נתיב המפעיל)
;   - להחליף בקובץ ה-worker תבניות {#tag} לפי ערכי JSON
;   - להריץ את ה-worker ולהמתין
;   - לאתר את הקובץ העדכני בתיקיית מקור (אופצ' לפי סיומת), להעביר ליעד
;   - לפתוח Gmail, לצרף את הקובץ העדכני מתיקייה מוגדרת, ולשלוח
; דרישות:
;   - AutoHotkey v2
;   - Edge/Chrome מחוברים ל-Gmail
;   - WinHTTP + ADODB זמינים להורדות
; קונפיג JSON:
;   - attachment.source (חובה)
;   - attachment.folder (חובה)
;   - attachment.file_pattern (אופצ.)
;   - mail2.to (אופצ.)    | mail2.subject (אופצ.)
;   - run_params (אופצ., מערך מחרוזות)
; ─────────────────────────────────────────────────────────

#Requires AutoHotkey v2.0
#SingleInstance Force

; ====== סביבה כללית ======
SetWorkingDir A_ScriptDir
SetTitleMatchMode 2
SetKeyDelay 50, 50
CoordMode "Mouse", "Screen"

; 
; (msg) {
;    msg := "Remark: " msg
;    TrayTip("Launcher", msg, 5)
;    OutputDebug(msg)
; }

; ====== קריאת קובץ הקונפיג של ה-launcher ======
cfgDir := A_ScriptDir "\\config"
launcherCfg := cfgDir "\\launcher.json"
if !FileExist(launcherCfg) {
    MsgBox "Missing launcher.json", "Launcher", 16
    ExitApp
}
launcherText := FileReadAll(launcherCfg)

BASE_URL   := JsonGet(launcherText, "base_url")
workerName := JsonGet(launcherText, "worker")
localDir   := JsonGet(launcherText, "local_dir")
if (localDir = "")
    dataDir := A_ScriptDir "\\RemoteAHK"
else if InStr(localDir, ":")
    dataDir := localDir
else
    dataDir := A_ScriptDir "\\" localDir

SCRIPT_URL    := BASE_URL workerName
workerJsonURL := BASE_URL StrReplace(workerName, ".ahk", ".json")

workerRaw := dataDir "\\" workerName
workerRun := dataDir "\\" StrReplace(workerName, ".ahk", ".resolved.ahk")
jsonPath  := dataDir "\\" StrReplace(workerName, ".ahk", ".json")
tmpScript := workerRaw ".tmp"
tmpJson   := jsonPath ".tmp"

; ערכי ברירת מחדל עבור כל ה-workers
defRunParams    := JsonGetArray(launcherText, "run_params")
defAttachSource := JsonGet(launcherText, "attachment.source")
defAttachFolder := JsonGet(launcherText, "attachment.folder")
defAttachPat    := JsonGet(launcherText, "attachment.file_pattern")
defMailTo       := JsonGet(launcherText, "mail2.to")
defMailSubject  := JsonGet(launcherText, "mail2.subject")

; ====== הכנת תיקיות ======
DirCreate dataDir

; ====== הורדת worker + בדיקת שינויים ======
;Notify("Downloading worker script...")
if !URLDownloadToFile2(SCRIPT_URL, tmpScript) {
    MsgBox "Failed to download worker.`n" SCRIPT_URL, "Launcher", 16
    Sleep 300000
    Reload
    ExitApp
}
if FileExist(workerRaw) {
    if (FileReadAll(workerRaw) = FileReadAll(tmpScript)) {
;        Notify("Worker script unchanged; waiting before retry")
        if FileExist(tmpScript)
            FileDelete tmpScript
        Sleep 300000
        Reload
        ExitApp
    }
}
FileMove tmpScript, workerRaw, 1
;Notify("Worker script ready")

; ====== הורדת JSON של ה-worker ======
;Notify("Downloading worker JSON...")
if URLDownloadToFile2(workerJsonURL, tmpJson) {
    FileMove tmpJson, jsonPath, 1
    ;Notify("Worker JSON downloaded")
} else {
    if FileExist(tmpJson)
        FileDelete tmpJson
    if FileExist(jsonPath)
        FileDelete jsonPath
    FileAppend "{}", jsonPath, "UTF-8"
    ;Notify("Worker JSON unavailable; using defaults")
}

; ====== מיזוג JSON עם ערכי ברירת מחדל ======
workerJson := FileReadAll(jsonPath)
runParamsArr := JsonGetArray(workerJson, "run_params")
if !runParamsArr.Length
    runParamsArr := defRunParams
attachSource := JsonGet(workerJson, "attachment.source")
if (attachSource = "") attachSource := defAttachSource
attachFolder := JsonGet(workerJson, "attachment.folder")
if (attachFolder = "") attachFolder := defAttachFolder
attachPat := JsonGet(workerJson, "attachment.file_pattern")
if (attachPat = "") attachPat := defAttachPat
mailTo := JsonGet(workerJson, "mail2.to")
if (mailTo = "") mailTo := defMailTo
mailSubject := JsonGet(workerJson, "mail2.subject")
if (mailSubject = "") mailSubject := defMailSubject

; ====== date fields (defaults if missing) ======
dateFrom := JsonGet(workerJson, "date_from")
dateTo   := JsonGet(workerJson, "date_to")
dateTag  := JsonGet(workerJson, "date")
if (dateFrom = "") {
    curMonth := FormatTime(A_Now, "MM")
    curYear  := FormatTime(A_Now, "yyyy")
    dateFrom := "01" curMonth curYear
}
if (dateTo = "")
    dateTo := FormatTime(A_Now, "ddMMyyyy")
if (dateTag = "")
    dateTag := FormatTime(A_Now, "ddMMyyyy")

mergedJson :=
(
    '{'
    . '"run_params":' ArrayToJson(runParamsArr) ','
    . '"attachment":{"source":"' attachSource '","folder":"' attachFolder '","file_pattern":"' attachPat '"},'
    . '"mail2":{"to":"' mailTo '","subject":"' mailSubject '"},'
    . '"date_from":"' dateFrom '",'
    . '"date_to":"' dateTo '",'
    . '"date":"' dateTag '"'
    . '}'
)

if FileExist(jsonPath)
    FileDelete jsonPath
FileAppend mergedJson, jsonPath, "UTF-8"
;Notify("Merged worker config")

; ====== קריאה והחתמה של נתוני הפעלה ב-JSON ======
oldJson := ""
try {
    oldJson := FileRead(jsonPath, "UTF-8")
} catch {
    oldJson := ""
}
if (oldJson = "")
    oldJson := "{}"

actName := StrReplace(A_ScriptName, '"', '\"')
actPath := StrReplace(A_ScriptFullPath, '"', '\"')

; עוטפים את ה-JSON המקורי תחת config + שדות הפעלה ברמה העליונה
newJson :=
(
    '{'
    . '"activated_by_name":"' actName '",'
    . '"activated_by_path":"' actPath '",'
    . '"reactivate_script_path":"' actPath '",'
    . '"reactivate_args":"",'
    . '"config":' oldJson
    . '}'
)
if FileExist(jsonPath)
    FileDelete jsonPath
FileAppend newJson, jsonPath, "UTF-8"

; ====== החלפת תגיות {#tag} בקובץ ה-worker ======
workerSrc := FileReadAll(workerRaw)
jsonText  := FileReadAll(jsonPath)

if (workerSrc = "") {
    MsgBox "Empty worker file.", "Launcher", 16
    ExitApp
}

workerResolved := ReplaceTags(workerSrc, jsonText)
if FileExist(workerRun)
    FileDelete workerRun
FileAppend workerResolved, workerRun, "UTF-8"
;Notify("Prepared worker script")

; ====== פרמטרים להרצת ה-worker ======
runParams := JsonGet(jsonText, "run_params_joined")
if (runParams = "")
    runParams := BuildRunParams(jsonText)

; ====== הרצת ה-worker (המתנה לסיום) ======
;Notify("Running worker...")
try {
    RunWait A_AhkPath ' "' workerRun '" ' runParams, dataDir
} catch as e {
    MsgBox "Failed to run worker:`n" e.Message, "Launcher", 16
    ExitApp
}
;Notify("Worker run complete")

; ====== איתור קובץ עדכני והעברה לתיקיית יעד ======
sourceDir := JsonGet(jsonText, "attachment.source")
outFolder := JsonGet(jsonText, "attachment.folder")
filePat   := JsonGet(jsonText, "attachment.file_pattern")

if (sourceDir = "" || outFolder = "") {
    MsgBox "Attachment config missing: attachment.source and/or attachment.folder", "Launcher", 16
    ExitApp
}

ext := GetExtensionFromPattern(filePat)       ; לדוגמה ".ods" או ריק
wild := (ext != "" ? "*" ext : "*")           ; תיקון: ברירת מחדל לכל קובץ, או לפי סיומת

latestPath := LatestFile(sourceDir, wild)
if (latestPath != "") {
    DirCreate outFolder
    destPath := outFolder "\" SplitPathName(latestPath)
    try
        FileMove(latestPath, destPath, 1)
    catch
    {
        ; ממשיכים גם אם לא הצליח להזיז
    }
    ;Notify("Moved latest file to " destPath)

}

; ====== שליחה ב-Gmail: פתיחת Compose + צירוף האחרון מתיקייה קבועה ======
toEmail := JsonGet(jsonText, "mail2.to")
subject := JsonGet(jsonText, "mail2.subject")
if (toEmail = "")  toEmail := "roger@seach.co.il"
if (subject = "")  subject := "Automated Report"

; *** עדכן כאן את התיקייה ממנה תצורף הקובץ ל-Gmail ***
attachFolder := "C:\Users\User\Documents\AutoHotkey\yarpareports"

latestAttachPath := LatestFile(attachFolder, "*")
if (latestAttachPath = "" || !FileExist(latestAttachPath)) {
    MsgBox "No files to attach in:`n" attachFolder, "Launcher", 16
    ExitApp
}
latestAttachName := SplitPathName(latestAttachPath)

composeUrl := "https://mail.google.com/mail/?view=cm&fs=1&tf=1"
;Notify("Opening Gmail compose window...")
if !OpenGmailCompose(composeUrl) {
    MsgBox "Couldn't open or focus a Gmail compose window.", "Launcher", 48
    ExitApp
}

; השהייה כדי לאפשר ל-Gmail להיטען
Sleep 10000

; אל כתובת הנמען
SendText(toEmail)
Sleep 300

; טאב → טאב כדי להגיע לנושא
Send "{Tab}"
Sleep 150
Send "{Tab}"
Sleep 200

; נושא
SendText(subject)
Sleep 200

; לגוף ההודעה (טאב אחד)
Send "{Tab}"
Sleep 150
SendText("Hi Boss, attached is the report.")
Sleep 300

; לכפתור הסיכה (בד"כ 3 טאבים מהגוף – מותאם לסביבה שלך)
Loop 4 {
    Send "{Tab}"
    Sleep 120
}
Send "{Enter}"

; המתנה לדיאלוג פתיחה
if !WinWaitActive("Open",, 4) && !WinWaitActive("File Upload",, 4) {
    Send "^o"
    if !WinWaitActive("Open",, 4) && !WinWaitActive("File Upload",, 4) {
        MsgBox "Couldn't open the file chooser to attach.", "Launcher", 48
        ExitApp
    }
}

; ניווט לשורת הנתיב, הדבקת התיקייה, אנטר
Send "!d"
Sleep 150
SendText(attachFolder)
Sleep 150
Send "{Enter}"
Sleep 400

; הקלדת שם הקובץ בלבד ואנטר לצירוף
SendText(latestAttachName)
Sleep 200
Send "{Enter}"

; ממתינים להעלאה (כוון לפי גודל קובץ/רשת)
Sleep 8000

; שליחה (Ctrl+Enter)
Send "^{Enter}"
Sleep 1200

;Notify("Email sent. Waiting before restart")
Sleep 300000
Reload
ExitApp

; ─────────────────────────────────────────────────────────
; Helpers
; ─────────────────────────────────────────────────────────

; --- דפדפן & Gmail ---
OpenGmailCompose(url) {
    if !WinExist("ahk_exe msedge.exe") && !WinExist("ahk_exe chrome.exe") {
        if TryRun('msedge.exe --new-window "' url '"') {
            if !WinWait("ahk_exe msedge.exe",, 10)
                return false
        } else if TryRun('chrome.exe --new-window "' url '"') {
            if !WinWait("ahk_exe chrome.exe",, 10)
                return false
        } else {
            Run url
        }
    }
    winSpec := ""
    if WinExist("ahk_exe msedge.exe")
        winSpec := "ahk_exe msedge.exe"
    else if WinExist("ahk_exe chrome.exe")
        winSpec := "ahk_exe chrome.exe"

    if (winSpec != "") {
        WinActivate winSpec
        if !WinWaitActive(winSpec,, 8)
            return false
        Sleep 250
        Send "^l"
        Sleep 120
        SendText(url)
        Sleep 120
        Send "{Enter}"
    }

    if !WinWait("Gmail",, 12) {
        if (winSpec != "") {
            WinActivate winSpec
            Sleep 250
            Send "^l"
            Sleep 120
            SendText(url)
            Sleep 120
            Send "{Enter}"
            if !WinWait("Gmail",, 8)
                return false
        } else {
            return false
        }
    }
    WinActivate "Gmail"
    Sleep 800
    return true
}

TryRun(cmd) {
    try {
        Run cmd
        return true
    } catch {
        return false
    }
}

; --- הורדה לקובץ ---
URLDownloadToFile2(url, outPath) {
    whr := ComObject("WinHttp.WinHttpRequest.5.1")
    try {
        whr.Open("GET", url, false)
        whr.Send()
        if (whr.Status != 200)
            return false
        ado := ComObject("ADODB.Stream")
        ado.Type := 1
        ado.Open()
        ado.Write(whr.ResponseBody)
        ado.SaveToFile(outPath, 2)
        ado.Close()
        return true
    } catch {
        return false
    }
}

FileReadAll(path) {
    try {
        return FileRead(path, "UTF-8")
    } catch {
        return ""
    }
}

; --- החלפת תגיות {#key} מתוך JSON (בודק גם config.key) ---
ReplaceTags(src, json) {
    out := ""
    pos := 1
    L := StrLen(src)
    while (pos <= L) {
        if RegExMatch(src, "\{#([\w\.\-]+)\}", &m, pos) {
            out .= SubStr(src, pos, m.Pos[0] - pos)
            key := m[1]
            val := JsonGet(json, key)
            if (val = "")
                val := JsonGet(json, "config." key)
            out .= val
            pos := m.Pos[0] + m.Len[0]
        } else {
            out .= SubStr(src, pos)
            break
        }
    }
    return out
}

; --- איתור קובץ עדכני לפי תו־כללי (ברירת מחדל "*", או "*.ods" וכו') ---
LatestFile(dir, wildcard := "*") {
    newest := ""
    newTime := 0
    Loop Files dir "\" wildcard, "F" {
        t := A_LoopFileTimeModified
        if (t > newTime) {
            newTime := t
            newest := A_LoopFileFullPath
        }
    }
    return newest
}

; --- שליפת סיומת מתבנית שם (למשל ".ods") ---
GetExtensionFromPattern(pat) {
    if (pat = "")
        return ""
    if RegExMatch(pat, "\.([A-Za-z0-9]+)$", &m)
        return "." m[1]
    return ""
}

SplitPathName(fp) {
    SplitPath fp, &nameOut
    return nameOut
}

; ── JSON helpers (פשוטים; מחרוזות ומערכים של מחרוזות) ──
JsonGet(j, keyPath) {
    parts := StrSplit(keyPath, ".")
    if (parts.Length = 1) {
        return JsonGetFlat(j, parts[1])
    } else {
        objText := JsonGetObjectText(j, parts[1])
        if (objText = "")
            objText := JsonGetObjectText(j, "config")
        if (objText != "")
            return JsonGetFlat(objText, parts[2])
    }

    if (keyPath = "run_params_joined") {
        arr := JsonGetArray(j, "run_params")
        if (arr.Length)
            return JoinWithSpaces(arr)
    }
    return ""
}

; מחפש מפתח "key": "value" בתוך טקסט JSON (ללא נסטינג)
JsonGetFlat(jt, key) {
    if (jt = "" || key = "")
        return ""
    ; תומך ברווחים וערכים טקסטואליים (ללא גרשיים פנימיים)
    pat := '"' key '"\s*:\s*"([^"]*)"'
    if RegExMatch(jt, pat, &m)
        return m[1]
    return ""
}

; מחלץ אובייקט לפי "key": { ... } (מאזן סוגריים מסולסלים)
JsonGetObjectText(jt, key) {
    if (jt = "" || key = "")
        return ""
    pat := '"' key '"\s*:\s*\{'
    pos := RegExMatch(jt, pat, &m)
    if !pos
        return ""
    start := m.Pos[0] + m.Len[0] - 1
    depth := 1, i := start + 1, L := StrLen(jt)
    while (i <= L) {
        ch := SubStr(jt, i, 1)
        if (ch = "{")
            depth++
        else if (ch = "}") {
            depth--
            if (depth = 0)
                return "{" . SubStr(jt, start+1, i-start-1) . "}"
        }
        i++
    }
    return ""
}

; קורא מערך מחרוזות: "key": ["a","b","c"]
JsonGetArray(jt, key) {
    arr := []
    if (jt = "" || key = "")
        return arr
    if RegExMatch(jt, '"' key '"\s*:\s*\[(.*?)\]', &m) {
        inner := m[1]
        ; פיצול על פסיקים ברמה שטוחה (פשוט; מניח שאין פסיקים בתוך מחרוזות)
        for token in StrSplit(inner, ",") {
            token := Trim(token)
            if RegExMatch(token, '^"(.*)"$', &mm) {
                ; מחליף \" כפולים פנימיים אם קיימים
                val := StrReplace(mm[1], '\"', '"')
                arr.Push(val)
            }
        }
    }
    return arr
}

BuildRunParams(jt) {
    a := JsonGetArray(jt, "run_params")
    if !a.Length
        return ""
    return JoinWithSpaces(a)
}

JoinWithSpaces(arr) {
    out := ""
    for i, v in arr
        out .= (i>1 ? " " : "") '"' v '"'
    return out
}

ArrayToJson(arr) {
    out := "["
    for i, v in arr
        out .= (i>1 ? "," : "") '"' StrReplace(v, '"', '\\"') '"'
    return out "]"
}
