; ─────────────────────────────────────────────────────────
; GeneralYarpaReport.ahk  (AutoHotkey v2)
; Generic Yarpa worker driven by per-report JSON config
; ─────────────────────────────────────────────────────────

#Requires AutoHotkey v2.0
#SingleInstance Force
#Warn
SetTitleMatchMode(2)
CoordMode("Mouse", "Screen")
SetDefaultMouseSpeed(0)
SetKeyDelay(50, 50)
SetWorkingDir(A_ScriptDir)

; ====== DEFAULT WAIT TIMES (ms) ======
WAIT_TINY  := 200
WAIT_SHORT := 600
WAIT_MED   := 1200
WAIT_LONG  := 3000
WAIT_LOAD  := 10000
WAIT_EXCEL_Y := 5000
WAIT_FINAL := 5000
DATE_SEP   := "/"
TYPE_ONE_DIGIT_MONTH := true

; ====== READ CONFIG JSON ======
cfgPath := A_Args.Length ? A_Args[1] : A_ScriptDir "\\" RegExReplace(A_ScriptName, "(?:\.resolved)?\.ahk$", ".json")
cfgJson := FileExist(cfgPath) ? FileRead(cfgPath, "UTF-8") : "{}"

; helper to override numeric or text setting
OverrideSetting(name) {
    global %name%, cfgJson
    val := JsonValue(cfgJson, name)
    if (val != "") {
        if name in DATE_SEP,RUN_CMD,SRC_DIR,DST_DIR,FILE_PATTERN
            %name% := val
        else if name = "TYPE_ONE_DIGIT_MONTH"
            TYPE_ONE_DIGIT_MONTH := (val = "true" || val = "1")
        else
            %name% := val + 0
    }
}

for k in ["WAIT_TINY","WAIT_SHORT","WAIT_MED","WAIT_LONG","WAIT_LOAD","WAIT_EXCEL_Y","WAIT_FINAL","DATE_SEP","RUN_CMD","SRC_DIR","DST_DIR","FILE_PATTERN","TYPE_ONE_DIGIT_MONTH"]
    OverrideSetting(k)

; coordinates
FROM_FIELD_X := JsonValue(cfgJson, "FROM_FIELD_X") + 0
FROM_FIELD_Y := JsonValue(cfgJson, "FROM_FIELD_Y") + 0
SUBFIELD_X   := JsonValue(cfgJson, "SUBFIELD_X") + 0
SUBFIELD_Y   := JsonValue(cfgJson, "SUBFIELD_Y") + 0
CONFIRM_X    := JsonValue(cfgJson, "CONFIRM_X") + 0
CONFIRM_Y    := JsonValue(cfgJson, "CONFIRM_Y") + 0
EXCEL_X      := JsonValue(cfgJson, "EXCEL_X") + 0
EXCEL_Y      := JsonValue(cfgJson, "EXCEL_Y") + 0

if (SRC_DIR = "")
    SRC_DIR := "C:\\psoftw\\doc"
if (DST_DIR = "")
    DST_DIR := "C:\\Users\\User\\Documents\\AutoHotkey\\yarpareports"
DirCreate(DST_DIR)

; ====== MAIN ======
try {
    ; 0) Desktop and launch Yarpa
    Send("#d")
    Sleep(8000)
    if (RUN_CMD != "") {
        Run(RUN_CMD, "C:\\psoftw")
        Sleep(WAIT_LOAD)
        Send("{Enter}")
        Sleep(WAIT_LOAD)
    }

    ; 1) Navigate according to sequence
    steps := JsonGetArray(cfgJson, "sequence")
    for step in steps {
        parts := StrSplit(step, ":")
        cmd := parts[1]
        if (cmd = "Send") {
            Send(parts[2])
            Sleep(GetWait(parts.Length >= 3 ? parts[3] : ""))
        } else if (cmd = "Sleep") {
            Sleep(GetWait(parts[2]))
        } else if (cmd = "SetDate") {
            SetDate(parts[2])
        }
    }

    ; 2) Confirm
    ClickAt(CONFIRM_X, CONFIRM_Y)
    Sleep(WAIT_MED)

    ; 3) Export
    exportStart := A_Now
    ClickAt(EXCEL_X, EXCEL_Y)
    Sleep(WAIT_EXCEL_Y)
    Send("Y")

    ; 4) Wait for newest exported file
    patterns := ["*.ods","*.xlsx","*.xls"]
    latestFile := WaitForNewestStableFile(SRC_DIR, patterns, exportStart, 120000, 800, 3)
    if (latestFile = "")
        throw Error("Timed out waiting for exported file in: " SRC_DIR)

    ; 5) Move → DST with name pattern
    SplitPath(latestFile, , , &spExt)
    base := FILE_PATTERN = "" ? "Report_" dateTag : StrReplace(FILE_PATTERN, "{date}", dateTag)
    destPath := UniquePath(DST_DIR "\\" base "." spExt)
    if !IsFileUnlocked(latestFile, 3, 500)
        throw Error("File is locked by another process: " latestFile)
    FileMove(latestFile, destPath, 1)

    ; 6) Log success
    FileAppend(FormatTime(A_Now, "yyyy-MM-dd HH:mm:ss") " OK`n", A_ScriptDir "\\worker.log", "UTF-8")
    Sleep(WAIT_FINAL)
} catch as e {
    MsgBox("ERROR: " e.Message, "GeneralYarpaReport", 16)
}
ExitApp()

; ====== HELPERS ======
GetWait(name) {
    global WAIT_TINY, WAIT_SHORT, WAIT_MED, WAIT_LONG
    if (name = "WAIT_TINY") return WAIT_TINY
    if (name = "WAIT_SHORT") return WAIT_SHORT
    if (name = "WAIT_MED") return WAIT_MED
    if (name = "WAIT_LONG") return WAIT_LONG
    if RegExMatch(name, "^\\d+$")
        return name + 0
    return 0
}

SetDate(mode) {
    global FROM_FIELD_X, FROM_FIELD_Y, SUBFIELD_X, SUBFIELD_Y, WAIT_SHORT, WAIT_TINY, DATE_SEP, TYPE_ONE_DIGIT_MONTH, dateTag
    if (mode = "FirstOfCurrent-Today") {
        MM := FormatTime(A_Now, "MM")
        YYYY := FormatTime(A_Now, "yyyy")
        from := "01" MM YYYY
        to   := FormatTime(A_Now, "ddMMyyyy")
        dateTag := YYYY "-" MM
        ClickAt(FROM_FIELD_X, FROM_FIELD_Y)
        Sleep(WAIT_SHORT)
        ClickAt(SUBFIELD_X, SUBFIELD_Y)
        Sleep(WAIT_TINY)
        TypeDateIntoActiveField(from, DATE_SEP)
        Send("{Tab}")
        Sleep(WAIT_TINY)
        TypeDateIntoActiveField(to, DATE_SEP)
    } else if (mode = "LastMonth") {
        prev := DateAdd(A_Now, -1, "M")
        MM := FormatTime(prev, "MM")
        YYYY := FormatTime(prev, "yyyy")
        from := "01" MM YYYY
        firstNext := DateAdd(prev, 1, "M")
        lastDate := DateAdd(firstNext, -1, "D")
        to := FormatTime(lastDate, "ddMMyyyy")
        dateTag := YYYY "-" MM
        ClickAt(FROM_FIELD_X, FROM_FIELD_Y)
        Sleep(WAIT_SHORT)
        ClickAt(SUBFIELD_X, SUBFIELD_Y)
        Sleep(WAIT_TINY)
        TypeDateIntoActiveField(from, DATE_SEP)
        Send("{Tab}")
        Sleep(WAIT_TINY)
        TypeDateIntoActiveField(to, DATE_SEP)
    } else if (mode = "PrevMonth") {
        prev := DateAdd(A_Now, -1, "M")
        YYYY := FormatTime(prev, "yyyy")
        MM2 := FormatTime(prev, "MM")
        MM1 := FormatTime(prev, "M")
        dateTag := YYYY "-" MM2
        monthToType := TYPE_ONE_DIGIT_MONTH ? MM1 : MM2
        ClickAt(FROM_FIELD_X, FROM_FIELD_Y)
        Sleep(WAIT_SHORT)
        Send("{Tab 3}")
        Sleep(40)
        Send("^a")
        Sleep(60)
        Send("{Home}")
        Sleep(40)
        Send(monthToType)
        Sleep(WAIT_SHORT)
    }
}

ClickAt(x, y) {
    MouseMove(x, y, 0)
    Sleep(80)
    Click()
}

TypeDateIntoActiveField(ddMMyyyy, sep := "/") {
    if (StrLen(ddMMyyyy) != 8) {
        onlyDigits := RegExReplace(ddMMyyyy, "\D")
        if (StrLen(onlyDigits) = 8)
            ddMMyyyy := onlyDigits
        else
            throw Error("Invalid date text: " ddMMyyyy)
    }
    d := SubStr(ddMMyyyy, 1, 2)
    m := SubStr(ddMMyyyy, 3, 2)
    y := SubStr(ddMMyyyy, 5, 4)
    Send("^a")
    Sleep(40)
    Send("{Home}")
    Sleep(20)
    Send(d . sep . m . sep . y)
    Sleep(80)
}

WaitForNewestStableFile(dirPath, patterns, sinceTime, timeoutMs := 60000, pollMs := 500, stableChecks := 2) {
    deadline := A_TickCount + timeoutMs
    newest := ""
    newestTime := ""
    while (A_TickCount < deadline) {
        newest := ""
        newestTime := ""
        for pat in patterns {
            Loop Files, dirPath "\\" pat, "F" {
                t := A_LoopFileTimeModified
                if (t >= sinceTime) {
                    if (newest = "" || t > newestTime) {
                        newest := A_LoopFileFullPath
                        newestTime := t
                    }
                }
            }
        }
        if (newest != "" && IsFileStable(newest, stableChecks, pollMs) && IsFileUnlocked(newest, 1, pollMs))
            return newest
        Sleep(pollMs)
    }
    return ""
}

IsFileStable(path, checks := 2, waitMs := 500) {
    if !FileExist(path)
        return false
    prev := FileGetSize(path, "B")
    loop checks {
        Sleep(waitMs)
        cur := FileGetSize(path, "B")
        if (cur != prev)
            return false
        prev := cur
    }
    return true
}

IsFileUnlocked(path, tries := 2, waitMs := 400) {
    loop tries {
        try {
            f := FileOpen(path, "rw")
            if (f) {
                f.Close()
                return true
            }
        } catch {
        }
        Sleep(waitMs)
    }
    return false
}

UniquePath(path) {
    if !FileExist(path)
        return path
    SplitPath(path, &uFile, &uDir, &uExt, &uName)
    i := 2
    loop {
        cand := uDir "\\" uName " (" i ")." uExt
        if !FileExist(cand)
            return cand
        i++
    }
}

JsonValue(json, key) {
    if RegExMatch(json, '"' key '"\s*:\s*"?([^",}\n]*)', &m)
        return m[1]
    return ""
}

JsonGetArray(json, key) {
    arr := []
    if RegExMatch(json, '"' key '"\s*:\s*\[(.*?)\]', &m) {
        inner := m[1]
        for token in StrSplit(inner, ",") {
            token := Trim(token)
            if RegExMatch(token, '^"(.*)"$', &mm)
                arr.Push(mm[1])
        }
    }
    return arr
}

