; ─────────────────────────────────────────────────────────
; evenhensales.ahk  (AutoHotkey v2) — Yarpa worker (robust)
; ─────────────────────────────────────────────────────────

#Requires AutoHotkey v2.0
#SingleInstance Force
#Warn
SetTitleMatchMode(2)
CoordMode("Mouse", "Screen")
SetDefaultMouseSpeed(0)
SetKeyDelay(50, 50)
SetWorkingDir(A_ScriptDir)

; ====== COORDINATES (SCREEN) ======
FROM_FIELD_X := 507 , FROM_FIELD_Y := 185
SUBFIELD_X   := 365 , SUBFIELD_Y   := 135
CONFIRM_X    := 602 , CONFIRM_Y    := 707
EXCEL_X      := 357 , EXCEL_Y      := 67

; ====== FILE PATHS ======
SRC_DIR := "C:\psoftw\doc"
DST_DIR := "C:\Users\User\Documents\AutoHotkey\yarpareports"
DirCreate(DST_DIR)

; ====== TIMINGS (ms) ======
WAIT_TINY       := 200
WAIT_SHORT      := 600
WAIT_MED        := 1200
WAIT_LONG       := 3000
WAIT_DESKTOP    := 1200     ; after Win+D
WAIT_LOAD       := 10000    ; extra load after launching
WAIT_EXCEL_Y    := 5000     ; wait before sending Y after Excel click
WAIT_FINAL      := 5000     ; FINAL extra sleep at the end

; ====== DATE SETTINGS ======
DATE_SEP := "/"   ; change if Yarpa expects '-' instead

; ====== MAIN ======
try {
    ; 0) Desktop
    Send("#d")
    Sleep(8000)

    ; 1) Open Yarpa (run program directly)
    Run("C:\\psoftw\\piryonS.exe", "C:\\psoftw")
    Sleep(WAIT_LOAD)
     Send("{Enter}")
    Sleep(WAIT_LOAD)

    ; 2) Navigate to the report
    Send("U")
    Sleep(WAIT_SHORT)
    Send("R")
    Sleep(WAIT_SHORT)
    Send("K")
    Sleep(WAIT_SHORT)
    Send("B")
    Sleep(WAIT_LONG)

    ; 3) Compute/fetch dates (from JSON or defaults)
    dateFrom := "{#date_from}"
    dateTo   := "{#date_to}"
    dateTag  := "{#date}"

    cfgPath := A_ScriptDir "\\" RegExReplace(A_ScriptName, "(?:\.resolved)?\.ahk$", ".json")
    if FileExist(cfgPath) {
        cfgJson := FileRead(cfgPath, "UTF-8")
        tmp := JsonValue(cfgJson, "date_from")
        if (dateFrom = "" || InStr(dateFrom, "{#")) && tmp != ""
            dateFrom := tmp
        tmp := JsonValue(cfgJson, "date_to")
        if (dateTo = "" || InStr(dateTo, "{#")) && tmp != ""
            dateTo := tmp
        tmp := JsonValue(cfgJson, "date")
        if (dateTag = "" || InStr(dateTag, "{#")) && tmp != ""
            dateTag := tmp
    }

    ; If JSON left them blank, set first-of-month -> today
    if (dateFrom = "" || InStr(dateFrom, "{#")) {
        MM   := FormatTime(A_Now, "MM")
        YYYY := FormatTime(A_Now, "yyyy")
        dateFrom := "01" MM YYYY          ; ddMMyyyy (no separators)
    }
    if (dateTo = "" || InStr(dateTo, "{#")) {
        dateTo := FormatTime(A_Now, "ddMMyyyy")  ; ddMMyyyy (no separators)
    }
    if (dateTag = "" || InStr(dateTag, "{#")) {
        dateTag := SubStr(dateFrom, 5, 4) "-" SubStr(dateFrom, 3, 2) ; yyyy-MM
    }

    ; 3a) Enter FROM date safely (with separators)
    ClickAt(FROM_FIELD_X, FROM_FIELD_Y)
    Sleep(WAIT_SHORT)
    ClickAt(SUBFIELD_X,   SUBFIELD_Y)
    Sleep(WAIT_TINY)
    TypeDateIntoActiveField(dateFrom, DATE_SEP)

    ; 3b) Move to TO field and enter safely (with separators)
    Send("{Tab}")
    Sleep(WAIT_TINY)
    TypeDateIntoActiveField(dateTo, DATE_SEP)

    ; 4) Confirm
    ClickAt(CONFIRM_X, CONFIRM_Y)
    Sleep(WAIT_MED)

    ; 5) Export → Excel icon
    exportStart := A_Now
    ClickAt(EXCEL_X, EXCEL_Y)

    ; Some exports prompt "Y" (overwrite/confirm)
    Sleep(WAIT_EXCEL_Y)
    Send("Y")

    ; 6) Wait for newest exported file (created after exportStart) to be stable & unlocked
    patterns := ["*.ods","*.xlsx","*.xls"]
    latestFile := WaitForNewestStableFile(SRC_DIR, patterns, exportStart, 120000, 800, 3)
    if (latestFile = "")
        throw Error("Timed out waiting for exported file in: " SRC_DIR)

    ; 7) Move → DST with nice name (ensure unique)
    SplitPath(latestFile, , , &spExt)
    outName  := "SalesReport_" dateTag "_Evenhen." spExt
    destPath := UniquePath(DST_DIR "\" outName)

    ; Final safety: ensure file is still free right before moving
    if !IsFileUnlocked(latestFile, 3, 500)
        throw Error("File is locked by another process: " latestFile)

    FileMove(latestFile, destPath, 1)

    ; 8) Log success (no MsgBox)
    FileAppend(FormatTime(A_Now, "yyyy-MM-dd HH:mm:ss") " OK`n", A_ScriptDir "\worker.log", "UTF-8")

    ; 9) FINAL: extra 5s sleep at the very end
    Sleep(WAIT_FINAL)
} catch as e {
    MsgBox("ERROR: " e.Message, "evenhensales", 16)
}
ExitApp()

; ====== HELPERS ======
ClickAt(x, y) {
    MouseMove(x, y, 0)
    Sleep(80)
    Click()
}

; Type an 8-digit ddMMyyyy string into a multi-part date control safely
; by forcing caret to the start and using separators so the control parses the full date.
TypeDateIntoActiveField(ddMMyyyy, sep := "/") {
    if (StrLen(ddMMyyyy) != 8) {
        ; Try to sanitize if user gave e.g. "01/09/2025"
        onlyDigits := RegExReplace(ddMMyyyy, "\D")
        if (StrLen(onlyDigits) = 8)
            ddMMyyyy := onlyDigits
        else
            throw Error("Invalid date text: " ddMMyyyy)
    }
    d := SubStr(ddMMyyyy, 1, 2)
    m := SubStr(ddMMyyyy, 3, 2)
    y := SubStr(ddMMyyyy, 5, 4)

    ; Make sure entire field is replaced, not one sub-slot
    Send("^a")
    Sleep(40)
    Send("{Home}")              ; caret at start of field
    Sleep(20)
    Send(d . sep . m . sep . y) ; e.g., 01/09/2025
    Sleep(80)
}

; Returns the newest file (matching any pattern) whose ModifiedTime is >= sinceTime
; and that becomes stable (size not changing) and unlocked, within timeoutMs.
WaitForNewestStableFile(dirPath, patterns, sinceTime, timeoutMs := 60000, pollMs := 500, stableChecks := 2) {
    deadline := A_TickCount + timeoutMs
    newest := ""
    newestTime := ""
    while (A_TickCount < deadline) {
        newest := ""
        newestTime := ""
        for pat in patterns {
            Loop Files, dirPath "\" pat, "F" {
                t := A_LoopFileTimeModified
                if (t >= sinceTime) {
                    if (newest = "" || t > newestTime) {
                        newest := A_LoopFileFullPath
                        newestTime := t
                    }
                }
            }
        }
        if (newest != "" && IsFileStable(newest, stableChecks, pollMs) && IsFileUnlocked(newest, 1, pollMs)) {
            return newest
        }
        Sleep(pollMs)
    }
    return ""
}

; Checks if file size stops changing across 'checks' intervals.
IsFileStable(path, checks := 2, waitMs := 500) {
    if !FileExist(path)
        return false
    prev := FileGetSize(path, "B")
    loop checks {
        Sleep(waitMs)
        cur := FileGetSize(path, "B")
        if (cur != prev)
            return false
        prev := cur
    }
    return true
}

; Attempts to open for read/write to detect sharing lock.
IsFileUnlocked(path, tries := 2, waitMs := 400) {
    loop tries {
        try {
            f := FileOpen(path, "rw")
            if (f) {
                f.Close()
                return true
            }
        } catch {
            ; locked - retry
        }
        Sleep(waitMs)
    }
    return false
}

; Keeps your original naming behavior but guarantees uniqueness.
UniquePath(path) {
    if !FileExist(path)
        return path
    SplitPath(path, &uFile, &uDir, &uExt, &uName)
    i := 2
    loop {
        cand := uDir "\" uName " (" i ")." uExt
        if !FileExist(cand)
            return cand
        i++
    }
}

JsonValue(json, key) {
    if RegExMatch(json, '"' key '"\s*:\s*"([^"]*)"', &m)
        return m[1]
    return ""
}
