; ─────────────────────────────────────────────────────────
; shefazer.ahk  (AutoHotkey v2) — Yarpa Report Z (robust month entry)
; ─────────────────────────────────────────────────────────

#Requires AutoHotkey v2.0
#SingleInstance Force
#Warn
SetTitleMatchMode(2)
CoordMode("Mouse", "Screen")
SetDefaultMouseSpeed(0)
SetKeyDelay(50, 50)
SetWorkingDir(A_ScriptDir)

; ====== COORDINATES ======
PATH_MONTH_SELECT := [770, 220]
BTN_CONFIRM       := [830, 700]
BTN_EXPORT_XLS    := [357, 67]

; ====== WAITS (ms) ======
WAIT_TINY     := 200
WAIT_SHORT    := 600
WAIT_MED      := 1200
WAIT_LONG     := 2500
WAIT_HUGE     := 8000
WAIT_LOAD     := 10000
WAIT_DOWNLOAD := 60000
WAIT_EXCEL_Y  := 5000
WAIT_FINAL    := 5000

; ====== SETTINGS ======
; If the month input control wants single-digit numbers (e.g., "8"), keep this true.
; If it needs two digits ("08"), set to false.
TYPE_ONE_DIGIT_MONTH := true

; ====== PATHS & JSON ======
jsonPath := A_ScriptDir "\\payload.json"
jsonText := "{}"
if FileExist(jsonPath)
    jsonText := FileRead(jsonPath, "UTF-8")
sourcePath := JsonGet(jsonText, "attachment.source")
if (sourcePath = "")
    sourcePath := "C:\\psoftw\\doc"
targetPath := JsonGet(jsonText, "attachment.folder")
if (targetPath = "")
    targetPath := "C:\\Users\\User\\Documents\\AutoHotkey\\yarpareports"
DirCreate(targetPath)

; ====== MAIN ======
try {
    ; 0) Determine month (from JSON or default prev month)
    dateTag := "{#date}"
    if (dateTag = "" || InStr(dateTag, "{#")) {
        info := GetPrevMonthInfo()
        prevMonth2 := info.month2                 ; "08"
        prevMonth1 := LTrim(info.month2, "0")     ; "8"
        prevStamp  := info.year "-" info.month2   ; "2025-08"
    } else if RegExMatch(dateTag, "^(\d{4})-(\d{2})$", &match) {
        prevMonth2 := match[2]
        prevMonth1 := LTrim(match[2], "0")
        prevStamp  := match[1] "-" match[2]
    } else if RegExMatch(dateTag, "^(\d{2})(\d{2})(\d{4})$", &match) {
        ; ddMMyyyy → use MM
        prevMonth2 := match[2]
        prevMonth1 := LTrim(match[2], "0")
        prevStamp  := match[3] "-" match[2]
    } else {
        throw Error("Invalid date text in JSON 'date': " dateTag)
    }

    monthToType := TYPE_ONE_DIGIT_MONTH ? (prevMonth1 = "" ? "0" : prevMonth1) : prevMonth2
    newFileName := "Report Z - " prevStamp

    ; 1) Yarpa flow
    Send("#d")
    Sleep(WAIT_HUGE)
    Run("C:\\psoftw\\piryonS.exe PIRYON7281.UDL", "C:\\psoftw")
    Sleep(WAIT_LOAD)
    Send("{Enter}")
    Sleep(WAIT_LOAD)

    Send("k")
    Sleep(WAIT_SHORT)
    Send("r")
    Sleep(WAIT_LONG)

    ; 2) Select month field, tab 3x & type safely
    Click(PATH_MONTH_SELECT[1], PATH_MONTH_SELECT[2])
    Sleep(WAIT_SHORT)
    Send("{Tab 3}")
    Sleep(40)
    ; Some screens jump through sub-fields — force a clean overwrite:
    Send("^a")
    Sleep(60)
    Send("{Home}")
    Sleep(40)
    Send(monthToType)
    Sleep(WAIT_SHORT)

    ; 3) Confirm
    Click(BTN_CONFIRM[1], BTN_CONFIRM[2])
    Sleep(WAIT_HUGE)

    ; 4) Export → Excel
    exportStart := A_Now
    Click(BTN_EXPORT_XLS[1], BTN_EXPORT_XLS[2])
    Sleep(WAIT_EXCEL_Y)
    Send("y")

    ; 5) Wait for newest exported file
    patterns := ["*.ods","*.xlsx","*.xls"]
    latestFile := WaitForNewestStableFile(sourcePath, patterns, exportStart, WAIT_DOWNLOAD, 800, 3)
    if (latestFile = "")
        throw Error("Timed out waiting for exported file in: " sourcePath)

    ; 6) Move/rename
    SplitPath(latestFile, , , &ext)
    newFilePath := UniquePath(targetPath "\\" newFileName "." ext)
    ; Safety: ensure unlocked before move
    if !IsFileUnlocked(latestFile, 3, 500)
        throw Error("File locked: " latestFile)
    FileMove(latestFile, newFilePath, 1)

    ; Log success
    FileAppend(FormatTime(A_Now, "yyyy-MM-dd HH:mm:ss") " OK`n", A_ScriptDir "\\worker.log", "UTF-8")
    Sleep(WAIT_FINAL)
} catch as e {
    MsgBox("ERROR: " e.Message, "shefazer", 16)
}
ExitApp()

; ====== HELPERS ======
GetPrevMonthInfo() {
    local y := Integer(FormatTime(A_Now, "yyyy"))
    local m := Integer(FormatTime(A_Now, "MM"))
    if (m = 1) {
        y -= 1, m := 12
    } else m -= 1
    return { year: y, monthNum: m, month2: Format("{:02}", m) }
}

WaitForNewestStableFile(dirPath, patterns, sinceTime, timeoutMs := 60000, pollMs := 500, stableChecks := 2) {
    deadline := A_TickCount + timeoutMs
    newest := ""
    newestTime := ""
    while (A_TickCount < deadline) {
        newest := ""
        newestTime := ""
        for pat in patterns {
            Loop Files, dirPath "\\" pat, "F" {
                t := A_LoopFileTimeModified
                if (t >= sinceTime) {
                    if (newest = "" || t > newestTime) {
                        newest := A_LoopFileFullPath
                        newestTime := t
                    }
                }
            }
        }
        if (newest != "" && IsFileStable(newest, stableChecks, pollMs) && IsFileUnlocked(newest, 1, pollMs)) {
            return newest
        }
        Sleep(pollMs)
    }
    return ""
}

IsFileStable(path, checks := 2, waitMs := 500) {
    if !FileExist(path)
        return false
    prev := FileGetSize(path, "B")
    loop checks {
        Sleep(waitMs)
        cur := FileGetSize(path, "B")
        if (cur != prev)
            return false
        prev := cur
    }
    return true
}

IsFileUnlocked(path, tries := 2, waitMs := 400) {
    loop tries {
        try {
            f := FileOpen(path, "rw")
            if (f) {
                f.Close()
                return true
            }
        } catch {
            ; locked - retry
        }
        Sleep(waitMs)
    }
    return false
}

UniquePath(path) {
    if !FileExist(path)
        return path
    SplitPath(path, &uFile, &uDir, &uExt, &uName)
    i := 2
    loop {
        cand := uDir "\\" uName " (" i ")." uExt
        if !FileExist(cand)
            return cand
        i++
    }
}

JsonGet(j, keyPath) {
    parts := StrSplit(keyPath, ".")
    if (parts.Length = 1)
        return JsonGetFlat(j, parts[1])
    objText := JsonGetObjectText(j, parts[1])
    if (objText != "")
        return JsonGetFlat(objText, parts[2])
    return ""
}

JsonGetFlat(jt, key) {
    if (jt = "" || key = "")
        return ""
    pat := '"' key '"\s*:\s*"([^"]*)"'
    if RegExMatch(jt, pat, &m)
        return m[1]
    return ""
}

JsonGetObjectText(jt, key) {
    if (jt = "" || key = "")
        return ""
    pat := '"' key '"\s*:\s*\{'
    pos := RegExMatch(jt, pat, &m)
    if !pos
        return ""
    start := m.Pos[0] + m.Len[0] - 1
    depth := 1, i := start + 1, L := StrLen(jt)
    while (i <= L) {
        ch := SubStr(jt, i, 1)
        if (ch = "{")
            depth++
        else if (ch = "}") {
            depth--
            if (depth = 0)
                return "{" . SubStr(jt, start+1, i-start-1) . "}"
        }
        i++
    }
    return ""
}
